# R-function for co-variation matrix
# The following R-function can be used to compute the proposed co-variation
# function (PIP).


###### function mvariationv2 #################
## Computes the variation matrix
## of a compositional (n,d) data matrix.
## No zeros allowed (not checked)
## There are five options of normalization:
## optnor="none",  variation matrix as is, [v_ij];
## optnor="minassoc" variation over minimal
##        association matrix;
## optnor="fstar", (PIP) rational 0,1 transform
## opnot="fa1", 1-exponential 0,1 transform (Aitchison1)
## opnor="fa2", exponential 0,1 transform (Aitchison2)
## These normalizations are (d number of parts)
## "minassoc" : t_ij = (d-1) v_ij / 2 totvar
## "fstar" : t_ij = 1 / [1 + sqrt(v_ij)]
## "fa1":   t_ij = 1 - exp( - 1 / sqrt(v_ij) )
## "fa2":   t_ij = exp(-sqrt(v_ij))
## The value of v_ij which distributes uniformly total
## variance over non-null entries is v_u = 2 totvar/(d-1)
## characteristic values
##  "no normalization"   "minassoc"    "fstar"     "fa1"     "fa2"
##     0                    0             1          1         1
##     1                    ...           0.5       0.632    0.368
##     v_u                  1             ...       ...       ...
##     Inf                  Inf           0         0          0
## fstar, fa1 and fa2 do not depend on the chosen subcomposition 
#########################################################
# JJ Egozcue, May 2017
# new options "fstar","fa1", "fa2", October 2022, March 2023
#########################################################
mvariationv2 <- function(x,optnor=c("none","minassoc","fstar","fa1","fa2")){
  # x is not checked for positiveness and matrix
  d = ncol(x); n = nrow(x)
  # compute clr 
  xlog = log(x)
  rmean = rowMeans(xlog)
  xclr = xlog - outer(rmean,rep(1,d))
  # variation matrix
  co = var(xclr)
  va = diag(co)
  co1 = matrix(rep(va,each=d),ncol = d)
  co2 = matrix(rep(va,times=d),ncol = d)
  varia = -2 * co + co1 + co2
  if(optnor=="none"){  return(varia) }
  if(optnor=="fstar"){ 
    fstar=1/(1+sqrt(varia))
    return(fstar)}
  if(optnor=="fa1"){
    fa1=1-exp(-1/sqrt(varia))
    return(fa1)}
  if(optnor=="fa2"){
    fa2=exp(-sqrt(varia))
    return(fa2)
  }
  #  option "minassoc" using totvar
  if(optnor=="minassoc"){
    totvar = sum(varia)/(2*d)
    varian= varia*((d-1)/(2*totvar))
    return(varian)
  }
}
##################################################

# The following R-script can be used to download data and compute the example
# used for illustration in Table 1.
library(robCompositions)
Adul = efsa[(efsa$Pop.Class=="Adults"),c(3:18,22)]
colnames(Adul)
# for the following example
fstar = mvariationv2(Adul[,1:11],optnor="fstar")
round(fstar,2)

# Output of the previous script
# Table:  $f^∗_N$ , for consumption of grains, vegetables, roots, nuts, 
# fruits, fish, meat, milk, eggs, sugar, fat. 
# In general linear association is weak. The maximum PIP is 0.80 for grains and
# sugar. This is not a strong linear association.

#     grn  veg  roo  nts  frt  mea  fsh  mlk  egg  sug  fat
#grn  1.00 0.67 0.70 0.60 0.76 0.75 0.63 0.71 0.59 0.80 0.68
#veg  0.67 1.00 0.63 0.66 0.71 0.75 0.63 0.62 0.69 0.65 0.77
#roo  0.70 0.63 1.00 0.57 0.63 0.73 0.57 0.68 0.62 0.71 0.65
#nts  0.60 0.66 0.57 1.00 0.61 0.64 0.59 0.59 0.63 0.59 0.63
#frt  0.76 0.71 0.63 0.61 1.00 0.73 0.65 0.68 0.63 0.71 0.72
#mea  0.75 0.75 0.73 0.64 0.73 1.00 0.61 0.68 0.66 0.73 0.77
#fish 0.63 0.63 0.57 0.59 0.65 0.61 1.00 0.62 0.60 0.57 0.60
#mlk  0.71 0.62 0.68 0.59 0.68 0.68 0.62 1.00 0.58 0.70 0.65
#egg  0.59 0.69 0.62 0.63 0.63 0.66 0.60 0.58 1.00 0.58 0.70
#sug  0.80 0.65 0.71 0.59 0.71 0.73 0.57 0.70 0.58 1.00 0.68
#fat  0.68 0.77 0.65 0.63 0.72 0.77 0.60 0.65 0.70 0.68 1.00

